import { useState, useEffect } from 'react'
import { useParams, useNavigate } from 'react-router-dom'
import toast from 'react-hot-toast'
import { adminGetOrder, adminUpdateOrderStatus } from '../services/api'

const STATUS_OPTS = ['pending', 'processing', 'shipped', 'delivered', 'cancelled', 'refunded']
const PAYMENT_OPTS = ['pending', 'paid', 'failed', 'refunded']

const OrderDetail = () => {
  const { id } = useParams()
  const navigate = useNavigate()
  const [order, setOrder] = useState(null)
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)
  const [status, setStatus] = useState('')
  const [paymentStatus, setPaymentStatus] = useState('')

  useEffect(() => {
    adminGetOrder(id)
      .then(r => {
        setOrder(r.data)
        setStatus(r.data.status)
        setPaymentStatus(r.data.payment_status)
      })
      .catch(() => toast.error('Failed to load order'))
      .finally(() => setLoading(false))
  }, [id])

  const handleUpdate = async () => {
    setSaving(true)
    try {
      await adminUpdateOrderStatus(id, { status, payment_status: paymentStatus })
      toast.success('Order updated!')
      setOrder(o => ({ ...o, status, payment_status: paymentStatus }))
    } catch { toast.error('Update failed') }
    finally { setSaving(false) }
  }

  if (loading) return <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
  if (!order) return <div className="text-center py-5 text-muted">Order not found</div>

  return (
    <div>
      <div className="d-flex align-items-center gap-3 mb-4">
        <button className="btn btn-outline-secondary btn-sm no-print" onClick={() => navigate('/orders')}>
          <i className="bi bi-arrow-left"></i>
        </button>
        <div className="flex-grow-1">
          <h4 className="fw-bold mb-0">Order {order.order_number}</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>
            {new Date(order.created_at).toLocaleString()}
          </p>
        </div>
        <span className={`badge-status badge-${order.status} fs-6`}>{order.status}</span>
        <button className="btn btn-outline-secondary btn-sm no-print" onClick={() => window.print()} title="Print Invoice">
          <i className="bi bi-printer me-1"></i>Print
        </button>
      </div>

      <div className="row">
        {/* Left col */}
        <div className="col-lg-8">
          {/* Items */}
          <div className="page-card mb-4">
            <div className="page-card-header">
              <h5><i className="bi bi-bag me-2"></i>Order Items</h5>
            </div>
            <div className="table-responsive">
              <table className="admin-table">
                <thead>
                  <tr><th>Product</th><th>Price</th><th>Qty</th><th>Subtotal</th></tr>
                </thead>
                <tbody>
                  {order.items?.map(item => (
                    <tr key={item.id}>
                      <td>
                        <div className="d-flex align-items-center gap-3">
                          <img
                            src={item.product?.primary_image?.url || item.product?.images?.[0]?.url || 'https://via.placeholder.com/44x44'}
                            style={{ width: 44, height: 44, objectFit: 'cover', borderRadius: 6, border: '1px solid #dee2e6' }}
                            onError={e => { e.target.src = 'https://via.placeholder.com/44x44' }}
                            alt=""
                          />
                          <div>
                            <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{item.product?.name || item.product_name}</div>
                            {item.variant && <div className="text-muted" style={{ fontSize: '.75rem' }}>{item.variant}</div>}
                          </div>
                        </div>
                      </td>
                      <td>${parseFloat(item.price).toFixed(2)}</td>
                      <td>{item.quantity}</td>
                      <td className="fw-semibold">${(item.price * item.quantity).toFixed(2)}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            <div className="p-3 border-top">
              <div className="d-flex justify-content-end">
                <div style={{ minWidth: 240 }}>
                  <div className="d-flex justify-content-between mb-1">
                    <span className="text-muted">Subtotal</span>
                    <span>${parseFloat(order.subtotal || order.total).toFixed(2)}</span>
                  </div>
                  {order.discount > 0 && (
                    <div className="d-flex justify-content-between mb-1 text-success">
                      <span>Discount</span>
                      <span>-${parseFloat(order.discount).toFixed(2)}</span>
                    </div>
                  )}
                  {order.shipping_cost > 0 && (
                    <div className="d-flex justify-content-between mb-1">
                      <span className="text-muted">Shipping</span>
                      <span>${parseFloat(order.shipping_cost).toFixed(2)}</span>
                    </div>
                  )}
                  <hr className="my-2" />
                  <div className="d-flex justify-content-between fw-bold">
                    <span>Total</span>
                    <span>${parseFloat(order.total).toFixed(2)}</span>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Shipping */}
          <div className="page-card mb-4">
            <div className="page-card-header">
              <h5><i className="bi bi-geo-alt me-2"></i>Shipping Address</h5>
            </div>
            <div className="p-4">
              <div className="fw-semibold">{order.shipping_name}</div>
              <div className="text-muted">{order.shipping_address}</div>
              <div className="text-muted">{order.shipping_city}{order.shipping_state ? `, ${order.shipping_state}` : ''} {order.shipping_zip}</div>
              <div className="text-muted">{order.shipping_country}</div>
              {order.shipping_phone && <div className="text-muted mt-1"><i className="bi bi-telephone me-1"></i>{order.shipping_phone}</div>}
            </div>
          </div>
        </div>

        {/* Right col */}
        <div className="col-lg-4">
          {/* Update Status */}
          <div className="form-card mb-4">
            <div className="form-section-title">Update Status</div>
            <div className="mb-3">
              <label className="form-label fw-semibold">Order Status</label>
              <select className="form-select" value={status} onChange={e => setStatus(e.target.value)}>
                {STATUS_OPTS.map(s => <option key={s} value={s}>{s.charAt(0).toUpperCase() + s.slice(1)}</option>)}
              </select>
            </div>
            <div className="mb-3">
              <label className="form-label fw-semibold">Payment Status</label>
              <select className="form-select" value={paymentStatus} onChange={e => setPaymentStatus(e.target.value)}>
                {PAYMENT_OPTS.map(s => <option key={s} value={s}>{s.charAt(0).toUpperCase() + s.slice(1)}</option>)}
              </select>
            </div>
            <button className="btn btn-primary w-100" onClick={handleUpdate} disabled={saving}>
              {saving ? <><span className="spinner-border spinner-border-sm me-2"></span>Updating...</> : 'Update Order'}
            </button>
          </div>

          {/* Customer */}
          <div className="form-card mb-4">
            <div className="form-section-title">Customer</div>
            {order.user ? (
              <div>
                <div className="fw-semibold">{order.user.name}</div>
                <div className="text-muted" style={{ fontSize: '.875rem' }}>{order.user.email}</div>
                <div className="text-muted" style={{ fontSize: '.875rem' }}>{order.user.phone}</div>
              </div>
            ) : (
              <div className="text-muted" style={{ fontSize: '.875rem' }}>Guest checkout</div>
            )}
          </div>

          {/* Payment */}
          <div className="form-card">
            <div className="form-section-title">Payment</div>
            <div className="d-flex justify-content-between mb-2">
              <span className="text-muted">Method</span>
              <span className="fw-semibold">{order.payment_method || '—'}</span>
            </div>
            <div className="d-flex justify-content-between mb-2">
              <span className="text-muted">Status</span>
              <span className={`badge-status badge-${order.payment_status}`}>{order.payment_status}</span>
            </div>
            {order.coupon_code && (
              <div className="d-flex justify-content-between">
                <span className="text-muted">Coupon</span>
                <span className="badge bg-success">{order.coupon_code}</span>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}
export default OrderDetail
