import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetNewsletter } from '../services/api'

const Newsletter = () => {
  const [subs, setSubs] = useState({ data: [], total: 0, last_page: 1 })
  const [filters, setFilters] = useState({ search: '', page: 1 })
  const [loading, setLoading] = useState(true)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetNewsletter(filters)
      setSubs(r.data)
    } catch { toast.error('Failed to load subscribers') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])

  const exportCSV = () => {
    const rows = (subs.data || []).map(s => `${s.email},${s.name || ''},${s.created_at}`)
    const csv = ['Email,Name,Subscribed At', ...rows].join('\n')
    const blob = new Blob([csv], { type: 'text/csv' })
    const url = URL.createObjectURL(blob)
    const a = document.createElement('a')
    a.href = url; a.download = 'newsletter-subscribers.csv'; a.click()
    URL.revokeObjectURL(url)
  }

  return (
    <div>
      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Newsletter Subscribers</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{subs.total} subscribers</p>
        </div>
        <button className="btn btn-outline-success" onClick={exportCSV} disabled={!subs.total}>
          <i className="bi bi-download me-2"></i>Export CSV
        </button>
      </div>

      {/* Stats card */}
      <div className="row mb-4">
        <div className="col-md-3">
          <div className="stat-card">
            <div className="stat-icon bg-primary bg-opacity-10">
              <i className="bi bi-megaphone text-primary"></i>
            </div>
            <div className="stat-body">
              <div className="stat-value">{subs.total}</div>
              <div className="stat-label">Total Subscribers</div>
            </div>
          </div>
        </div>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-6">
            <div style={{ position: 'relative' }}>
              <i className="bi bi-search" style={{ position: 'absolute', left: 12, top: '50%', transform: 'translateY(-50%)', color: '#adb5bd' }}></i>
              <input className="form-control" style={{ paddingLeft: 36 }} placeholder="Search by email..."
                value={filters.search} onChange={e => setFilters(f => ({ ...f, search: e.target.value, page: 1 }))} />
            </div>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ search: '', page: 1 })}>Reset</button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : (subs.data || []).length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-megaphone d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No subscribers yet</p>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>#</th><th>Email</th><th>Name</th><th>Subscribed</th></tr>
              </thead>
              <tbody>
                {(subs.data || []).map((s, i) => (
                  <tr key={s.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{((filters.page - 1) * 20) + i + 1}</td>
                    <td>
                      <a href={`mailto:${s.email}`} className="text-primary">{s.email}</a>
                    </td>
                    <td>{s.name || <span className="text-muted">—</span>}</td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{new Date(s.created_at).toLocaleDateString()}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
        {subs.last_page > 1 && (
          <div className="d-flex justify-content-center py-3">
            <div className="admin-pagination">
              <button disabled={filters.page <= 1} onClick={() => setFilters(f => ({ ...f, page: f.page - 1 }))}><i className="bi bi-chevron-left"></i></button>
              {Array.from({ length: subs.last_page }, (_, i) => i + 1).map(p => (
                <button key={p} className={filters.page === p ? 'active' : ''} onClick={() => setFilters(f => ({ ...f, page: p }))}>{p}</button>
              ))}
              <button disabled={filters.page >= subs.last_page} onClick={() => setFilters(f => ({ ...f, page: f.page + 1 }))}><i className="bi bi-chevron-right"></i></button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
export default Newsletter
