import { useState, useEffect } from 'react'
import { Link } from 'react-router-dom'
import toast from 'react-hot-toast'
import { adminGetSettings, adminUpdateSettings } from '../services/api'
import ImageUpload from '../components/ui/ImageUpload'

const DEFAULT_GALLERY = [
  { image: '', title: 'Gaming',      subtitle: 'Collection',          label: 'Hot Deal', link: '/shop' },
  { image: '', title: 'Controllers', subtitle: 'For all platforms',              link: '/shop' },
  { image: '', title: 'VR Headsets', subtitle: 'Experience the future',          link: '/shop' },
  { image: '', title: 'Accessories', subtitle: 'Must-have gadgets',              link: '/shop' },
  { image: '', title: 'Smartphones', subtitle: 'Latest models available',        link: '/shop' },
]

const DEFAULT_TRUST_BADGES = [
  { icon: '', title: 'Free Shipping',  desc: 'On orders over $50' },
  { icon: '', title: 'Easy Returns',   desc: '30-day return policy' },
  { icon: '', title: 'Secure Payment', desc: '100% secure checkout' },
  { icon: '', title: '24/7 Support',   desc: 'Dedicated support team' },
]
const DEFAULT_TESTIMONIALS = [
  { avatar: '', name: 'Sarah Johnson',  stars: 5, text: 'Amazing quality products and super fast shipping! Highly recommend!' },
  { avatar: '', name: 'Michael Brown',  stars: 5, text: 'Great customer service! They resolved my issue within hours.' },
  { avatar: '', name: 'Emily Davis',    stars: 5, text: 'Best prices online! Found exactly what I needed in minutes.' },
]
const DEFAULT_BRANDS = [
  { image: '', name: 'Apple',    link: '/shop?brand=apple' },
  { image: '', name: 'Samsung',  link: '/shop?brand=samsung' },
  { image: '', name: 'Sony',     link: '/shop?brand=sony' },
  { image: '', name: 'Dell',     link: '/shop?brand=dell' },
  { image: '', name: 'HP',       link: '/shop?brand=hp' },
  { image: '', name: 'Logitech', link: '/shop?brand=logitech' },
]

const ensure = (val, def) => {
  if (!val) return def
  if (Array.isArray(val)) return val
  try { return JSON.parse(val) } catch { return def }
}

// ─── Section header ───────────────────────────────────────────
const SectionHead = ({ icon, color, title, desc, onAdd, addLabel }) => (
  <div className="d-flex align-items-center justify-content-between mb-3">
    <div className="d-flex align-items-center gap-2">
      <div style={{ width: 36, height: 36, borderRadius: 8, background: `var(--bs-${color}-bg-subtle, #f0f4ff)`, display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
        <i className={`bi ${icon} text-${color}`}></i>
      </div>
      <div>
        <div className="fw-bold" style={{ fontSize: '.9rem' }}>{title}</div>
        {desc && <div className="text-muted" style={{ fontSize: '.75rem' }}>{desc}</div>}
      </div>
    </div>
    {onAdd && (
      <button className="btn btn-sm btn-outline-primary" onClick={onAdd}>
        <i className="bi bi-plus me-1"></i>{addLabel || 'Add'}
      </button>
    )}
  </div>
)

// ─── Main component ───────────────────────────────────────────
const HomePage = () => {
  const [loading, setLoading]       = useState(true)
  const [saving, setSaving]         = useState(false)
  const [gallery, setGallery]             = useState(DEFAULT_GALLERY)
  const [trustBadges, setTrustBadges]     = useState(DEFAULT_TRUST_BADGES)
  const [testimonials, setTestimonials]   = useState(DEFAULT_TESTIMONIALS)
  const [brands, setBrands]               = useState(DEFAULT_BRANDS)

  useEffect(() => {
    adminGetSettings()
      .then(r => {
        const s = r.data
        setGallery(ensure(s.gallery_panels, DEFAULT_GALLERY))
        setTrustBadges(ensure(s.trust_badges, DEFAULT_TRUST_BADGES))
        setTestimonials(ensure(s.testimonials, DEFAULT_TESTIMONIALS))
        setBrands(ensure(s.brands, DEFAULT_BRANDS))
      })
      .catch(() => {})
      .finally(() => setLoading(false))
  }, [])

  const handleSave = async () => {
    setSaving(true)
    try {
      await adminUpdateSettings({ gallery_panels: gallery, trust_badges: trustBadges, testimonials, brands })
      toast.success('Home page saved!')
    } catch { toast.error('Save failed') }
    finally { setSaving(false) }
  }

  // Gallery helpers
  const setGalleryPanel = (i, k, v) => setGallery(arr => { const a = [...arr]; a[i] = { ...a[i], [k]: v }; return a })

  // Trust badge helpers
  const setBadge = (i, k, v) => setTrustBadges(arr => { const a = [...arr]; a[i] = { ...a[i], [k]: v }; return a })
  const addBadge = () => setTrustBadges(a => [...a, { icon: '', title: '', desc: '' }])
  const removeBadge = (i) => setTrustBadges(a => a.filter((_, idx) => idx !== i))

  // Testimonial helpers
  const setTestimonial = (i, k, v) => setTestimonials(arr => { const a = [...arr]; a[i] = { ...a[i], [k]: v }; return a })
  const addTestimonial = () => setTestimonials(a => [...a, { avatar: '', name: '', stars: 5, text: '' }])
  const removeTestimonial = (i) => setTestimonials(a => a.filter((_, idx) => idx !== i))

  // Brand helpers
  const setBrand = (i, k, v) => setBrands(arr => { const a = [...arr]; a[i] = { ...a[i], [k]: v }; return a })
  const addBrand = () => setBrands(a => [...a, { image: '', name: '', link: '' }])
  const removeBrand = (i) => setBrands(a => a.filter((_, idx) => idx !== i))

  if (loading) return <div className="loading-overlay"><div className="spinner-border text-primary"></div></div>

  return (
    <div>
      {/* Header */}
      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Home Page</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>Manage all sections of the home page</p>
        </div>
        <button className="btn btn-primary" onClick={handleSave} disabled={saving}>
          {saving
            ? <><span className="spinner-border spinner-border-sm me-2"></span>Saving...</>
            : <><i className="bi bi-check-circle me-2"></i>Save All Changes</>}
        </button>
      </div>

      {/* ── Quick Links ── */}
      <div className="row g-3 mb-4">
        {[
          {
            to: '/sliders', icon: 'bi-play-circle', color: 'primary',
            title: 'Hero Slider',
            desc: 'Main banner — add slides with image, title & button',
            tag: 'Top of page',
          },
          {
            to: '/banners?position=home_top', icon: 'bi-layout-sidebar-reverse', color: 'success',
            title: 'Side Banners',
            desc: 'Small banners shown to the right of the hero slider',
            tag: 'Next to hero',
          },
          {
            to: '/banners?position=home_bottom', icon: 'bi-image', color: 'warning',
            title: 'Bottom Banners',
            desc: 'Full-width banners shown below the product sections',
            tag: 'Below products',
          },
          {
            to: '/products', icon: 'bi-box-seam', color: 'info',
            title: 'Products',
            desc: 'Toggle Featured / Deal / New Arrival / Best Seller per product',
            tag: 'Product sections',
          },
          {
            to: '/categories', icon: 'bi-tag', color: 'secondary',
            title: 'Categories',
            desc: 'Edit category names and images shown in shop navigation',
            tag: 'Navigation',
          },
        ].map((item, i) => (
          <div key={i} className="col-md-6 col-xl-4">
            <Link to={item.to} className="text-decoration-none">
              <div className="page-card p-3 h-100" style={{ transition: 'box-shadow .2s', cursor: 'pointer' }}
                onMouseEnter={e => e.currentTarget.style.boxShadow = '0 4px 16px rgba(0,0,0,.1)'}
                onMouseLeave={e => e.currentTarget.style.boxShadow = 'none'}>
                <div className="d-flex align-items-start gap-3">
                  <div className={`bg-${item.color} bg-opacity-10 d-flex align-items-center justify-content-center flex-shrink-0`}
                    style={{ width: 42, height: 42, borderRadius: 10 }}>
                    <i className={`bi ${item.icon} text-${item.color}`} style={{ fontSize: '1.1rem' }}></i>
                  </div>
                  <div className="flex-grow-1 min-w-0">
                    <div className="d-flex align-items-center gap-2 mb-1">
                      <span className="fw-bold" style={{ fontSize: '.875rem' }}>{item.title}</span>
                      <span className={`badge bg-${item.color} bg-opacity-10 text-${item.color}`} style={{ fontSize: '.6rem', fontWeight: 600 }}>{item.tag}</span>
                    </div>
                    <div className="text-muted" style={{ fontSize: '.75rem', lineHeight: 1.5 }}>{item.desc}</div>
                  </div>
                </div>
                <div className="text-end mt-2">
                  <span className={`text-${item.color}`} style={{ fontSize: '.75rem', fontWeight: 600 }}>
                    Edit <i className="bi bi-arrow-right"></i>
                  </span>
                </div>
              </div>
            </Link>
          </div>
        ))}
      </div>

      {/* ── Gallery Section ── */}
      <div className="form-card mb-4">
        <SectionHead
          icon="bi-grid-3x3-gap" color="danger"
          title="Gallery Section"
          desc="5 promo panels shown below the hero banner (Gaming, Controllers, VR…)"
        />
        {(() => {
          const LABELS = ['Main Left Panel', 'Top-Right Big', 'Top-Right Small', 'Bottom-Right Small', 'Bottom-Right Big']
          return (
            <div className="row g-3">
              {gallery.map((panel, i) => (
                <div key={i} className={i === 0 ? 'col-12 col-lg-6' : 'col-md-6 col-xl-3'}>
                  <div className="p-3 border rounded" style={{ background: '#fff8f8' }}>
                    <div className="mb-2">
                      <span className="badge bg-danger bg-opacity-10 text-danger fw-semibold" style={{ fontSize: '.7rem' }}>
                        Panel {i + 1} — {LABELS[i]}
                      </span>
                    </div>
                    <label className="form-label fw-semibold" style={{ fontSize: '.75rem' }}>Image</label>
                    <ImageUpload value={panel.image} onChange={v => setGalleryPanel(i, 'image', v)} height={90} />
                    {i === 0 && (
                      <input className="form-control form-control-sm mt-2" placeholder='Label e.g. "Hot Deal"'
                        value={panel.label || ''} onChange={e => setGalleryPanel(i, 'label', e.target.value)} />
                    )}
                    <input className="form-control form-control-sm mt-2" placeholder="Title e.g. Gaming"
                      value={panel.title} onChange={e => setGalleryPanel(i, 'title', e.target.value)} />
                    <input className="form-control form-control-sm mt-1" placeholder="Subtitle e.g. Collection"
                      value={panel.subtitle || ''} onChange={e => setGalleryPanel(i, 'subtitle', e.target.value)} />
                    <input className="form-control form-control-sm mt-1" placeholder="Link e.g. /shop"
                      value={panel.link || ''} onChange={e => setGalleryPanel(i, 'link', e.target.value)} />
                  </div>
                </div>
              ))}
            </div>
          )
        })()}
        <small className="text-muted mt-3 d-block">
          <i className="bi bi-info-circle me-1"></i>
          Upload an image for each panel. If no image uploaded, the default template image is shown.
        </small>
      </div>

      {/* ── Trust Badges ── */}
      <div className="form-card mb-4">
        <SectionHead
          icon="bi-shield-check" color="success"
          title="Trust Badges"
          desc="4 icons shown below the hero banner (Free Shipping, Returns, etc.)"
          onAdd={addBadge} addLabel="Add Badge"
        />
        <div className="row g-3">
          {trustBadges.map((badge, i) => (
            <div key={i} className="col-xl-3 col-md-6">
              <div className="p-3 border rounded" style={{ background: '#f8fffe' }}>
                <div className="d-flex justify-content-between align-items-center mb-2">
                  <span className="badge bg-success bg-opacity-10 text-success fw-semibold" style={{ fontSize: '.7rem' }}>
                    Badge {i + 1}
                  </span>
                  <button className="btn btn-sm btn-outline-danger py-0 px-2" onClick={() => removeBadge(i)}>
                    <i className="bi bi-trash" style={{ fontSize: '.75rem' }}></i>
                  </button>
                </div>
                <label className="form-label fw-semibold" style={{ fontSize: '.75rem' }}>Icon Image</label>
                <ImageUpload value={badge.icon} onChange={v => setBadge(i, 'icon', v)} height={80} />
                <input className="form-control form-control-sm mt-2" placeholder="Title e.g. Free Shipping"
                  value={badge.title} onChange={e => setBadge(i, 'title', e.target.value)} />
                <input className="form-control form-control-sm mt-1" placeholder="Desc e.g. On orders over $50"
                  value={badge.desc} onChange={e => setBadge(i, 'desc', e.target.value)} />
              </div>
            </div>
          ))}
        </div>
        <small className="text-muted mt-3 d-block">
          <i className="bi bi-info-circle me-1"></i>
          Upload SVG or PNG icons. If no icon uploaded, default template icons will be used.
        </small>
      </div>

      {/* ── Testimonials ── */}
      <div className="form-card mb-4">
        <SectionHead
          icon="bi-chat-quote" color="primary"
          title="Customer Testimonials"
          desc="Reviews shown near the bottom of the home page"
          onAdd={addTestimonial} addLabel="Add Testimonial"
        />
        <div className="row g-3">
          {testimonials.map((t, i) => (
            <div key={i} className="col-xl-4 col-md-6">
              <div className="p-3 border rounded" style={{ background: '#f8f9ff' }}>
                <div className="d-flex justify-content-between align-items-center mb-2">
                  <span className="badge bg-primary bg-opacity-10 text-primary fw-semibold" style={{ fontSize: '.7rem' }}>
                    Review {i + 1}
                  </span>
                  <button className="btn btn-sm btn-outline-danger py-0 px-2" onClick={() => removeTestimonial(i)}>
                    <i className="bi bi-trash" style={{ fontSize: '.75rem' }}></i>
                  </button>
                </div>
                <label className="form-label fw-semibold" style={{ fontSize: '.75rem' }}>Customer Photo</label>
                <ImageUpload value={t.avatar} onChange={v => setTestimonial(i, 'avatar', v)} height={80} />
                <input className="form-control form-control-sm mt-2" placeholder="Customer Name"
                  value={t.name} onChange={e => setTestimonial(i, 'name', e.target.value)} />
                <select className="form-select form-select-sm mt-1" value={t.stars}
                  onChange={e => setTestimonial(i, 'stars', Number(e.target.value))}>
                  {[5, 4, 3, 2, 1].map(s => <option key={s} value={s}>{s} Star{s > 1 ? 's' : ''}</option>)}
                </select>
                <textarea className="form-control form-control-sm mt-1" rows={3} placeholder="Review text..."
                  value={t.text} onChange={e => setTestimonial(i, 'text', e.target.value)} />
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* ── Brand Logos ── */}
      <div className="form-card mb-4">
        <SectionHead
          icon="bi-award" color="warning"
          title="Brand Logos"
          desc="Brand logos shown in the 'Our Trusted Brands' section"
          onAdd={addBrand} addLabel="Add Brand"
        />
        <div className="row g-3">
          {brands.map((b, i) => (
            <div key={i} className="col-xl-2 col-md-3 col-sm-4 col-6">
              <div className="p-3 border rounded text-center" style={{ background: '#fffdf0' }}>
                <div className="d-flex justify-content-between align-items-center mb-2">
                  <span className="badge bg-warning bg-opacity-10 text-warning fw-semibold" style={{ fontSize: '.65rem' }}>
                    {i + 1}
                  </span>
                  <button className="btn btn-sm btn-outline-danger py-0 px-1" onClick={() => removeBrand(i)}>
                    <i className="bi bi-trash" style={{ fontSize: '.7rem' }}></i>
                  </button>
                </div>
                <ImageUpload value={b.image} onChange={v => setBrand(i, 'image', v)} height={65} />
                <input className="form-control form-control-sm mt-2" placeholder="Brand name"
                  value={b.name} onChange={e => setBrand(i, 'name', e.target.value)} />
                <input className="form-control form-control-sm mt-1" placeholder="Link /shop?brand=..."
                  value={b.link} onChange={e => setBrand(i, 'link', e.target.value)} />
              </div>
            </div>
          ))}
        </div>
        <small className="text-muted mt-3 d-block">
          <i className="bi bi-info-circle me-1"></i>
          If no logo uploaded, default brand images from the template will be used.
        </small>
      </div>

      {/* Save button bottom */}
      <div className="d-flex justify-content-end">
        <button className="btn btn-primary px-4" onClick={handleSave} disabled={saving}>
          {saving
            ? <><span className="spinner-border spinner-border-sm me-2"></span>Saving...</>
            : <><i className="bi bi-check-circle me-2"></i>Save All Changes</>}
        </button>
      </div>
    </div>
  )
}
export default HomePage
