import { useState, useEffect } from 'react'
import { Link } from 'react-router-dom'
import { getStats } from '../services/api'
import toast from 'react-hot-toast'

const StatCard = ({ icon, label, value, color, to }) => (
  <div className="col-sm-6 col-xl-3 mb-4">
    <Link to={to} className="text-decoration-none">
      <div className="stat-card">
        <div className={`stat-icon bg-${color} bg-opacity-10`}>
          <i className={`bi ${icon} text-${color}`}></i>
        </div>
        <div className="stat-body">
          <div className="stat-value">{value}</div>
          <div className="stat-label">{label}</div>
        </div>
      </div>
    </Link>
  </div>
)

const Dashboard = () => {
  const [stats, setStats] = useState({})
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    getStats()
      .then(r => setStats(r.data))
      .catch(() => toast.error('Failed to load stats'))
      .finally(() => setLoading(false))
  }, [])

  if (loading) return <div className="loading-overlay"><div className="spinner-border text-primary"></div></div>

  return (
    <div>
      <div className="mb-4">
        <h4 className="fw-bold mb-1">Dashboard</h4>
        <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>Welcome back! Here's what's happening.</p>
      </div>

      {/* Stat cards */}
      <div className="row">
        <StatCard icon="bi-currency-dollar" label="Total Revenue" value={`$${parseFloat(stats.total_revenue || 0).toFixed(2)}`} color="success" to="/orders" />
        <StatCard icon="bi-cart-check" label="Total Orders" value={stats.total_orders || 0} color="primary" to="/orders" />
        <StatCard icon="bi-people" label="Total Users" value={stats.total_users || 0} color="info" to="/users" />
        <StatCard icon="bi-box-seam" label="Total Products" value={stats.total_products || 0} color="warning" to="/products" />
      </div>

      {/* Alert rows */}
      <div className="row mb-4">
        {stats.pending_orders > 0 && (
          <div className="col-md-3 mb-3">
            <Link to="/orders?status=pending" className="text-decoration-none">
              <div className="page-card p-3 border-start border-warning border-4">
                <div className="d-flex align-items-center gap-3">
                  <i className="bi bi-hourglass-split text-warning fs-4"></i>
                  <div><div className="fw-bold">{stats.pending_orders}</div><small className="text-muted">Pending Orders</small></div>
                </div>
              </div>
            </Link>
          </div>
        )}
        {stats.pending_wholesale > 0 && (
          <div className="col-md-3 mb-3">
            <Link to="/wholesale" className="text-decoration-none">
              <div className="page-card p-3 border-start border-info border-4">
                <div className="d-flex align-items-center gap-3">
                  <i className="bi bi-building text-info fs-4"></i>
                  <div><div className="fw-bold">{stats.pending_wholesale}</div><small className="text-muted">Wholesale Apps</small></div>
                </div>
              </div>
            </Link>
          </div>
        )}
        {stats.unread_contacts > 0 && (
          <div className="col-md-3 mb-3">
            <Link to="/contacts" className="text-decoration-none">
              <div className="page-card p-3 border-start border-danger border-4">
                <div className="d-flex align-items-center gap-3">
                  <i className="bi bi-envelope text-danger fs-4"></i>
                  <div><div className="fw-bold">{stats.unread_contacts}</div><small className="text-muted">Unread Messages</small></div>
                </div>
              </div>
            </Link>
          </div>
        )}
        {stats.pending_reviews > 0 && (
          <div className="col-md-3 mb-3">
            <Link to="/reviews" className="text-decoration-none">
              <div className="page-card p-3 border-start border-secondary border-4">
                <div className="d-flex align-items-center gap-3">
                  <i className="bi bi-star text-secondary fs-4"></i>
                  <div><div className="fw-bold">{stats.pending_reviews}</div><small className="text-muted">Pending Reviews</small></div>
                </div>
              </div>
            </Link>
          </div>
        )}
      </div>

      <div className="row">
        {/* Revenue Chart */}
        <div className="col-lg-7 mb-4">
          <div className="page-card h-100">
            <div className="page-card-header">
              <h5><i className="bi bi-bar-chart me-2 text-primary"></i>Revenue — Last 7 Days</h5>
            </div>
            <div className="p-4">
              {stats.revenue_chart && stats.revenue_chart.length > 0 ? (
                <div className="d-flex align-items-end gap-2" style={{ height: 160 }}>
                  {stats.revenue_chart.map((d, i) => {
                    const max = Math.max(...stats.revenue_chart.map(x => x.revenue), 1)
                    const h = Math.max((d.revenue / max) * 140, 4)
                    return (
                      <div key={i} className="d-flex flex-column align-items-center flex-grow-1">
                        <small className="text-muted mb-1" style={{ fontSize: '.65rem' }}>
                          {d.revenue > 0 ? '$' + Math.round(d.revenue) : ''}
                        </small>
                        <div style={{ height: h, background: '#0d6efd', borderRadius: 4, width: '100%', minHeight: 4 }}></div>
                        <small className="text-muted mt-1" style={{ fontSize: '.65rem' }}>{d.date}</small>
                      </div>
                    )
                  })}
                </div>
              ) : (
                <div className="text-center text-muted py-5">No revenue data yet</div>
              )}
            </div>
          </div>
        </div>

        {/* Top Products */}
        <div className="col-lg-5 mb-4">
          <div className="page-card h-100">
            <div className="page-card-header">
              <h5><i className="bi bi-trophy me-2 text-warning"></i>Top Products</h5>
            </div>
            <div className="p-0">
              {stats.top_products?.length > 0 ? stats.top_products.map((p, i) => (
                <div key={p.id} className="d-flex align-items-center gap-3 px-4 py-3 border-bottom">
                  <span className="badge bg-light text-dark fw-bold" style={{ width: 24 }}>{i + 1}</span>
                  <div className="flex-grow-1">
                    <div style={{ fontSize: '.875rem', fontWeight: 600 }}>{p.name}</div>
                    <small className="text-muted">{p.sales_count} sold · ${p.price}</small>
                  </div>
                  <div className="text-warning" style={{ fontSize: '.8rem' }}>
                    <i className="bi bi-star-fill me-1"></i>{p.rating || '—'}
                  </div>
                </div>
              )) : (
                <div className="text-center text-muted py-4">No sales data yet</div>
              )}
            </div>
          </div>
        </div>

        {/* Recent Orders */}
        <div className="col-12 mb-4">
          <div className="page-card">
            <div className="page-card-header">
              <h5><i className="bi bi-clock-history me-2 text-success"></i>Recent Orders</h5>
              <Link to="/orders" className="btn btn-sm btn-outline-primary">View All</Link>
            </div>
            {stats.recent_orders?.length > 0 ? (
              <div className="table-responsive">
                <table className="admin-table">
                  <thead>
                    <tr><th>Order #</th><th>Customer</th><th>Total</th><th>Status</th><th>Date</th></tr>
                  </thead>
                  <tbody>
                    {stats.recent_orders.map(o => (
                      <tr key={o.id}>
                        <td><Link to={`/orders/${o.id}`} className="text-primary fw-semibold">{o.order_number}</Link></td>
                        <td>{o.user?.name || o.shipping_name}</td>
                        <td className="fw-semibold">${parseFloat(o.total).toFixed(2)}</td>
                        <td><span className={`badge-status badge-${o.status}`}>{o.status}</span></td>
                        <td className="text-muted">{new Date(o.created_at).toLocaleDateString()}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            ) : (
              <div className="text-center text-muted py-4">No orders yet</div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}
export default Dashboard
