import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetCoupons, adminCreateCoupon, adminUpdateCoupon, adminDeleteCoupon } from '../services/api'
import Confirm from '../components/ui/Confirm'

const today = new Date().toISOString().split('T')[0]
const EMPTY = { code: '', type: 'percentage', value: '', min_order: '', max_discount: '', usage_limit: '', expires_at: '', is_active: true }

const Coupons = () => {
  const [coupons, setCoupons] = useState([])
  const [loading, setLoading] = useState(true)
  const [modal, setModal] = useState(false)
  const [form, setForm] = useState(EMPTY)
  const [editId, setEditId] = useState(null)
  const [saving, setSaving] = useState(false)
  const [confirm, setConfirm] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetCoupons()
      setCoupons(r.data)
    } catch { toast.error('Failed to load coupons') }
    finally { setLoading(false) }
  }, [])

  useEffect(() => { load() }, [load])

  const openCreate = () => { setForm(EMPTY); setEditId(null); setModal(true) }
  const openEdit = (c) => {
    setForm({
      code: c.code, type: c.type, value: c.value, min_order: c.min_order || '',
      max_discount: c.max_discount || '', usage_limit: c.usage_limit || '',
      expires_at: c.expires_at ? c.expires_at.split('T')[0] : '', is_active: !!c.is_active
    })
    setEditId(c.id)
    setModal(true)
  }

  const set = (k, v) => setForm(f => ({ ...f, [k]: v }))

  const generateCode = () => {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'
    const code = Array.from({ length: 8 }, () => chars[Math.floor(Math.random() * chars.length)]).join('')
    set('code', code)
  }

  const handleSubmit = async (e) => {
    e.preventDefault()
    setSaving(true)
    try {
      if (editId) await adminUpdateCoupon(editId, form)
      else await adminCreateCoupon(form)
      toast.success(editId ? 'Coupon updated!' : 'Coupon created!')
      setModal(false)
      load()
    } catch (err) {
      toast.error(err.response?.data?.message || 'Save failed')
    } finally { setSaving(false) }
  }

  const handleDelete = async (id) => {
    try {
      await adminDeleteCoupon(id)
      toast.success('Coupon deleted')
      load()
    } catch { toast.error('Delete failed') }
    setConfirm(null)
  }

  const isExpired = (c) => c.expires_at && new Date(c.expires_at) < new Date()

  return (
    <div>
      <Confirm show={!!confirm} message="Delete this coupon?" onConfirm={() => handleDelete(confirm)} onCancel={() => setConfirm(null)} />

      {modal && (
        <div className="modal d-block" style={{ background: 'rgba(0,0,0,.5)' }}>
          <div className="modal-dialog modal-dialog-centered">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">{editId ? 'Edit Coupon' : 'New Coupon'}</h5>
                <button className="btn-close" onClick={() => setModal(false)}></button>
              </div>
              <form onSubmit={handleSubmit}>
                <div className="modal-body">
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Coupon Code *</label>
                    <div className="input-group">
                      <input className="form-control text-uppercase" value={form.code} onChange={e => set('code', e.target.value.toUpperCase())} required />
                      <button type="button" className="btn btn-outline-secondary" onClick={generateCode} title="Generate random code">
                        <i className="bi bi-shuffle"></i>
                      </button>
                    </div>
                  </div>
                  <div className="row g-3 mb-3">
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Discount Type</label>
                      <select className="form-select" value={form.type} onChange={e => set('type', e.target.value)}>
                        <option value="percentage">Percentage (%)</option>
                        <option value="fixed">Fixed Amount ($)</option>
                      </select>
                    </div>
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Value *</label>
                      <div className="input-group">
                        <span className="input-group-text">{form.type === 'percentage' ? '%' : '$'}</span>
                        <input className="form-control" type="number" step="0.01" min="0" value={form.value} onChange={e => set('value', e.target.value)} required />
                      </div>
                    </div>
                  </div>
                  <div className="row g-3 mb-3">
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Minimum Order</label>
                      <div className="input-group">
                        <span className="input-group-text">$</span>
                        <input className="form-control" type="number" step="0.01" min="0" value={form.min_order} onChange={e => set('min_order', e.target.value)} placeholder="Optional" />
                      </div>
                    </div>
                    {form.type === 'percentage' && (
                      <div className="col-md-6">
                        <label className="form-label fw-semibold">Max Discount</label>
                        <div className="input-group">
                          <span className="input-group-text">$</span>
                          <input className="form-control" type="number" step="0.01" min="0" value={form.max_discount} onChange={e => set('max_discount', e.target.value)} placeholder="Optional" />
                        </div>
                      </div>
                    )}
                  </div>
                  <div className="row g-3 mb-3">
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Usage Limit</label>
                      <input className="form-control" type="number" min="1" value={form.usage_limit} onChange={e => set('usage_limit', e.target.value)} placeholder="Unlimited" />
                    </div>
                    <div className="col-md-6">
                      <label className="form-label fw-semibold">Expires At</label>
                      <input className="form-control" type="date" min={today} value={form.expires_at} onChange={e => set('expires_at', e.target.value)} />
                    </div>
                  </div>
                  <div className="form-check form-switch">
                    <input className="form-check-input" type="checkbox" checked={form.is_active} onChange={e => set('is_active', e.target.checked)} id="cp_active" />
                    <label className="form-check-label" htmlFor="cp_active">Active</label>
                  </div>
                </div>
                <div className="modal-footer">
                  <button type="button" className="btn btn-secondary" onClick={() => setModal(false)}>Cancel</button>
                  <button type="submit" className="btn btn-primary" disabled={saving}>
                    {saving ? <span className="spinner-border spinner-border-sm me-1"></span> : null}
                    {editId ? 'Update' : 'Create'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Coupons</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{coupons.length} coupons</p>
        </div>
        <button className="btn btn-primary" onClick={openCreate}>
          <i className="bi bi-plus-circle me-2"></i>Add Coupon
        </button>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : coupons.length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-ticket-perforated d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No coupons yet</p>
            <button className="btn btn-primary btn-sm" onClick={openCreate}>Add First Coupon</button>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>Code</th><th>Type</th><th>Value</th><th>Min Order</th><th>Usage</th><th>Expires</th><th>Status</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {coupons.map(c => (
                  <tr key={c.id}>
                    <td>
                      <span className="badge bg-dark text-white" style={{ fontFamily: 'monospace', fontSize: '.875rem', letterSpacing: 1 }}>
                        {c.code}
                      </span>
                    </td>
                    <td><span className="badge bg-light text-dark">{c.type}</span></td>
                    <td className="fw-semibold">{c.type === 'percentage' ? `${c.value}%` : `$${parseFloat(c.value).toFixed(2)}`}</td>
                    <td>{c.min_order ? `$${parseFloat(c.min_order).toFixed(2)}` : '—'}</td>
                    <td>
                      <span className="text-muted">{c.used_count || 0}</span>
                      {c.usage_limit ? <span className="text-muted"> / {c.usage_limit}</span> : <span className="text-muted"> / ∞</span>}
                    </td>
                    <td className={isExpired(c) ? 'text-danger' : 'text-muted'} style={{ fontSize: '.8rem' }}>
                      {c.expires_at ? new Date(c.expires_at).toLocaleDateString() : '—'}
                      {isExpired(c) && <div style={{ fontSize: '.7rem' }}>Expired</div>}
                    </td>
                    <td>
                      <span className={`badge-status badge-${c.is_active && !isExpired(c) ? 'active' : 'inactive'}`}>
                        {c.is_active && !isExpired(c) ? 'Active' : 'Inactive'}
                      </span>
                    </td>
                    <td>
                      <div className="action-btns">
                        <button className="btn btn-sm btn-outline-primary" onClick={() => openEdit(c)}><i className="bi bi-pencil"></i></button>
                        <button className="btn btn-sm btn-outline-danger" onClick={() => setConfirm(c.id)}><i className="bi bi-trash"></i></button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  )
}
export default Coupons
