import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetContacts, adminMarkContactRead } from '../services/api'

const Contacts = () => {
  const [contacts, setContacts] = useState({ data: [], total: 0, last_page: 1 })
  const [filters, setFilters] = useState({ status: '', page: 1 })
  const [loading, setLoading] = useState(true)
  const [detail, setDetail] = useState(null)
  const [marking, setMarking] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetContacts(filters)
      setContacts(r.data)
    } catch { toast.error('Failed to load messages') }
    finally { setLoading(false) }
  }, [filters])

  useEffect(() => { load() }, [load])

  const handleView = async (c) => {
    setDetail(c)
    if (!c.is_read) {
      setMarking(c.id)
      try {
        await adminMarkContactRead(c.id)
        load()
      } catch {}
      finally { setMarking(null) }
    }
  }

  return (
    <div>
      {/* Detail Modal */}
      {detail && (
        <div className="modal d-block" style={{ background: 'rgba(0,0,0,.5)' }}>
          <div className="modal-dialog modal-dialog-centered">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">{detail.subject || 'Message'}</h5>
                <button className="btn-close" onClick={() => setDetail(null)}></button>
              </div>
              <div className="modal-body">
                <div className="row g-2 mb-3">
                  <div className="col-md-6">
                    <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.75rem' }}>FROM</div>
                    <div style={{ fontWeight: 600 }}>{detail.name}</div>
                    <div className="text-primary">{detail.email}</div>
                    {detail.phone && <div className="text-muted">{detail.phone}</div>}
                  </div>
                  <div className="col-md-6">
                    <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.75rem' }}>RECEIVED</div>
                    <div>{new Date(detail.created_at).toLocaleString()}</div>
                  </div>
                </div>
                {detail.subject && (
                  <div className="mb-3">
                    <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.75rem' }}>SUBJECT</div>
                    <div>{detail.subject}</div>
                  </div>
                )}
                <div>
                  <div className="fw-semibold text-muted mb-1" style={{ fontSize: '.75rem' }}>MESSAGE</div>
                  <div className="p-3 bg-light rounded" style={{ whiteSpace: 'pre-wrap', fontSize: '.875rem' }}>{detail.message}</div>
                </div>
              </div>
              <div className="modal-footer">
                <a href={`mailto:${detail.email}?subject=Re: ${detail.subject || 'Your inquiry'}`} className="btn btn-primary">
                  <i className="bi bi-reply me-1"></i>Reply via Email
                </a>
                <button className="btn btn-secondary" onClick={() => setDetail(null)}>Close</button>
              </div>
            </div>
          </div>
        </div>
      )}

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Contact Messages</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{contacts.total} messages</p>
        </div>
      </div>

      <div className="page-card mb-4 p-3">
        <div className="row g-2">
          <div className="col-md-4">
            <select className="form-select" value={filters.status} onChange={e => setFilters(f => ({ ...f, status: e.target.value, page: 1 }))}>
              <option value="">All Messages</option>
              <option value="unread">Unread</option>
              <option value="read">Read</option>
            </select>
          </div>
          <div className="col-md-2">
            <button className="btn btn-outline-secondary w-100" onClick={() => setFilters({ status: '', page: 1 })}>Reset</button>
          </div>
        </div>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : (contacts.data || []).length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-envelope d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No contact messages</p>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>Name</th><th>Email</th><th>Subject</th><th>Status</th><th>Date</th><th>Action</th></tr>
              </thead>
              <tbody>
                {(contacts.data || []).map(c => (
                  <tr key={c.id} style={{ fontWeight: c.is_read ? 400 : 600 }}>
                    <td>{c.name}</td>
                    <td className="text-muted">{c.email}</td>
                    <td>
                      <div style={{ maxWidth: 200 }} className="text-truncate">
                        {c.subject || <span className="text-muted">(no subject)</span>}
                      </div>
                    </td>
                    <td>
                      {c.is_read
                        ? <span className="badge bg-light text-dark">Read</span>
                        : <span className="badge bg-primary">New</span>}
                    </td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{new Date(c.created_at).toLocaleDateString()}</td>
                    <td>
                      <button className="btn btn-sm btn-outline-primary" onClick={() => handleView(c)} disabled={marking === c.id}>
                        {marking === c.id ? <span className="spinner-border spinner-border-sm"></span> : <><i className="bi bi-eye me-1"></i>View</>}
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
        {contacts.last_page > 1 && (
          <div className="d-flex justify-content-center py-3">
            <div className="admin-pagination">
              <button disabled={filters.page <= 1} onClick={() => setFilters(f => ({ ...f, page: f.page - 1 }))}><i className="bi bi-chevron-left"></i></button>
              {Array.from({ length: contacts.last_page }, (_, i) => i + 1).map(p => (
                <button key={p} className={filters.page === p ? 'active' : ''} onClick={() => setFilters(f => ({ ...f, page: p }))}>{p}</button>
              ))}
              <button disabled={filters.page >= contacts.last_page} onClick={() => setFilters(f => ({ ...f, page: f.page + 1 }))}><i className="bi bi-chevron-right"></i></button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
export default Contacts
