import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetCategories, adminCreateCategory, adminUpdateCategory, adminDeleteCategory } from '../services/api'
import Confirm from '../components/ui/Confirm'
import ImageUpload from '../components/ui/ImageUpload'

const EMPTY = { name: '', slug: '', description: '', parent_id: '', image: '', is_active: true }

const Categories = () => {
  const [cats, setCats] = useState([])
  const [loading, setLoading] = useState(true)
  const [modal, setModal] = useState(null) // null | 'create' | 'edit'
  const [form, setForm] = useState(EMPTY)
  const [editId, setEditId] = useState(null)
  const [saving, setSaving] = useState(false)
  const [confirm, setConfirm] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetCategories()
      setCats(r.data)
    } catch { toast.error('Failed to load categories') }
    finally { setLoading(false) }
  }, [])

  useEffect(() => { load() }, [load])

  const openCreate = () => { setForm(EMPTY); setEditId(null); setModal('edit') }
  const openEdit = (c) => {
    setForm({ name: c.name, slug: c.slug || '', description: c.description || '', parent_id: c.parent_id || '', image: c.image || '', is_active: !!c.is_active })
    setEditId(c.id)
    setModal('edit')
  }

  const set = (k, v) => setForm(f => ({ ...f, [k]: v }))

  const handleSubmit = async (e) => {
    e.preventDefault()
    setSaving(true)
    try {
      if (editId) await adminUpdateCategory(editId, form)
      else await adminCreateCategory(form)
      toast.success(editId ? 'Category updated!' : 'Category created!')
      setModal(null)
      load()
    } catch (err) {
      toast.error(err.response?.data?.message || 'Save failed')
    } finally { setSaving(false) }
  }

  const handleDelete = async (id) => {
    try {
      await adminDeleteCategory(id)
      toast.success('Category deleted')
      load()
    } catch { toast.error('Delete failed') }
    setConfirm(null)
  }

  const parents = cats.filter(c => !c.parent_id)

  return (
    <div>
      <Confirm show={!!confirm} message="Delete this category?" onConfirm={() => handleDelete(confirm)} onCancel={() => setConfirm(null)} />

      {/* Modal */}
      {modal && (
        <div className="modal d-block" style={{ background: 'rgba(0,0,0,.5)' }}>
          <div className="modal-dialog modal-dialog-centered">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">{editId ? 'Edit Category' : 'New Category'}</h5>
                <button className="btn-close" onClick={() => setModal(null)}></button>
              </div>
              <form onSubmit={handleSubmit}>
                <div className="modal-body">
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Name *</label>
                    <input className="form-control" value={form.name} onChange={e => set('name', e.target.value)} required />
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Slug</label>
                    <input className="form-control" value={form.slug} onChange={e => set('slug', e.target.value)} placeholder="auto-generated if blank" />
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Parent Category</label>
                    <select className="form-select" value={form.parent_id} onChange={e => set('parent_id', e.target.value)}>
                      <option value="">None (top-level)</option>
                      {parents.filter(p => p.id !== editId).map(p => (
                        <option key={p.id} value={p.id}>{p.name}</option>
                      ))}
                    </select>
                  </div>
                  <div className="mb-3">
                    <ImageUpload value={form.image} onChange={v => set('image', v)} label="Image" height={110} />
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Description</label>
                    <textarea className="form-control" rows={2} value={form.description} onChange={e => set('description', e.target.value)} />
                  </div>
                  <div className="form-check form-switch">
                    <input className="form-check-input" type="checkbox" checked={form.is_active} onChange={e => set('is_active', e.target.checked)} id="cat_active" />
                    <label className="form-check-label" htmlFor="cat_active">Active</label>
                  </div>
                </div>
                <div className="modal-footer">
                  <button type="button" className="btn btn-secondary" onClick={() => setModal(null)}>Cancel</button>
                  <button type="submit" className="btn btn-primary" disabled={saving}>
                    {saving ? <span className="spinner-border spinner-border-sm me-1"></span> : null}
                    {editId ? 'Update' : 'Create'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Categories</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{cats.length} categories</p>
        </div>
        <button className="btn btn-primary" onClick={openCreate}>
          <i className="bi bi-plus-circle me-2"></i>Add Category
        </button>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : cats.length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-tag d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No categories yet</p>
            <button className="btn btn-primary btn-sm" onClick={openCreate}>Add First Category</button>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>#</th><th>Category</th><th>Parent</th><th>Products</th><th>Status</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {cats.map(c => (
                  <tr key={c.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{c.id}</td>
                    <td>
                      <div className="d-flex align-items-center gap-3">
                        {c.image && (
                          <img src={c.image} style={{ width: 36, height: 36, objectFit: 'cover', borderRadius: 6 }}
                            onError={e => { e.target.style.display = 'none' }} alt="" />
                        )}
                        <div>
                          <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{c.name}</div>
                          <div className="text-muted" style={{ fontSize: '.75rem' }}>{c.slug}</div>
                        </div>
                      </div>
                    </td>
                    <td>{c.parent ? <span className="badge bg-light text-dark">{c.parent.name}</span> : <span className="text-muted">—</span>}</td>
                    <td><span className="badge bg-light text-dark">{c.products_count ?? '—'}</span></td>
                    <td><span className={`badge-status badge-${c.is_active ? 'active' : 'inactive'}`}>{c.is_active ? 'Active' : 'Inactive'}</span></td>
                    <td>
                      <div className="action-btns">
                        <button className="btn btn-sm btn-outline-primary" onClick={() => openEdit(c)}><i className="bi bi-pencil"></i></button>
                        <button className="btn btn-sm btn-outline-danger" onClick={() => setConfirm(c.id)}><i className="bi bi-trash"></i></button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  )
}
export default Categories
