import { useState, useEffect, useCallback } from 'react'
import toast from 'react-hot-toast'
import { adminGetBanners, adminCreateBanner, adminUpdateBanner, adminDeleteBanner } from '../services/api'
import Confirm from '../components/ui/Confirm'
import ImageUpload from '../components/ui/ImageUpload'

const EMPTY = { title: '', subtitle: '', image: '', link: '', position: 'home_top', sort_order: 0, is_active: true }
const POSITIONS = [
  { value: 'home_top',    label: 'Home — Side Banners (next to hero slider, max 2)' },
  { value: 'home_middle', label: 'Home — Gallery Section (promo grid below hero)' },
  { value: 'home_bottom', label: 'Home — Bottom Banners (full-width, below products)' },
  { value: 'shop_top',    label: 'Shop Page — Top Banner' },
  { value: 'category',    label: 'Category Pages — Banner' },
]

const Banners = () => {
  const [banners, setBanners] = useState([])
  const [loading, setLoading] = useState(true)
  const [modal, setModal] = useState(false)
  const [form, setForm] = useState(EMPTY)
  const [editId, setEditId] = useState(null)
  const [saving, setSaving] = useState(false)
  const [confirm, setConfirm] = useState(null)

  const load = useCallback(async () => {
    setLoading(true)
    try {
      const r = await adminGetBanners()
      setBanners(r.data)
    } catch { toast.error('Failed to load banners') }
    finally { setLoading(false) }
  }, [])

  useEffect(() => { load() }, [load])

  const openCreate = () => { setForm(EMPTY); setEditId(null); setModal(true) }
  const openEdit = (b) => {
    setForm({ title: b.title || '', subtitle: b.subtitle || '', image: b.image || '', link: b.link || '', position: b.position || 'home_top', sort_order: b.sort_order || 0, is_active: !!b.is_active })
    setEditId(b.id)
    setModal(true)
  }

  const set = (k, v) => setForm(f => ({ ...f, [k]: v }))

  const handleSubmit = async (e) => {
    e.preventDefault()
    setSaving(true)
    try {
      if (editId) await adminUpdateBanner(editId, form)
      else await adminCreateBanner(form)
      toast.success(editId ? 'Banner updated!' : 'Banner created!')
      setModal(false)
      load()
    } catch (err) {
      toast.error(err.response?.data?.message || 'Save failed')
    } finally { setSaving(false) }
  }

  const handleDelete = async (id) => {
    try {
      await adminDeleteBanner(id)
      toast.success('Banner deleted')
      load()
    } catch { toast.error('Delete failed') }
    setConfirm(null)
  }

  return (
    <div>
      <Confirm show={!!confirm} message="Delete this banner?" onConfirm={() => handleDelete(confirm)} onCancel={() => setConfirm(null)} />

      {modal && (
        <div className="modal d-block" style={{ background: 'rgba(0,0,0,.5)' }}>
          <div className="modal-dialog modal-dialog-centered">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">{editId ? 'Edit Banner' : 'New Banner'}</h5>
                <button className="btn-close" onClick={() => setModal(false)}></button>
              </div>
              <form onSubmit={handleSubmit}>
                <div className="modal-body">
                  <div className="mb-3">
                    <ImageUpload value={form.image} onChange={v => set('image', v)} label="Image" required height={120} />
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Title</label>
                    <input className="form-control" value={form.title} onChange={e => set('title', e.target.value)} />
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Subtitle</label>
                    <input className="form-control" value={form.subtitle} onChange={e => set('subtitle', e.target.value)} />
                  </div>
                  <div className="row g-3 mb-3">
                    <div className="col-md-8">
                      <label className="form-label fw-semibold">Link URL</label>
                      <input className="form-control" value={form.link} onChange={e => set('link', e.target.value)} placeholder="/shop" />
                    </div>
                    <div className="col-md-4">
                      <label className="form-label fw-semibold">Sort Order</label>
                      <input className="form-control" type="number" min="0" value={form.sort_order} onChange={e => set('sort_order', parseInt(e.target.value) || 0)} />
                    </div>
                  </div>
                  <div className="mb-3">
                    <label className="form-label fw-semibold">Position</label>
                    <select className="form-select" value={form.position} onChange={e => set('position', e.target.value)}>
                      {POSITIONS.map(p => (
                        <option key={p.value} value={p.value}>{p.label}</option>
                      ))}
                    </select>
                  </div>
                  <div className="form-check form-switch">
                    <input className="form-check-input" type="checkbox" checked={form.is_active} onChange={e => set('is_active', e.target.checked)} id="bn_active" />
                    <label className="form-check-label" htmlFor="bn_active">Active</label>
                  </div>
                </div>
                <div className="modal-footer">
                  <button type="button" className="btn btn-secondary" onClick={() => setModal(false)}>Cancel</button>
                  <button type="submit" className="btn btn-primary" disabled={saving}>
                    {saving ? <span className="spinner-border spinner-border-sm me-1"></span> : null}
                    {editId ? 'Update' : 'Create'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}

      <div className="d-flex align-items-center justify-content-between mb-4">
        <div>
          <h4 className="fw-bold mb-1">Banners</h4>
          <p className="text-muted mb-0" style={{ fontSize: '.875rem' }}>{banners.length} banner{banners.length !== 1 ? 's' : ''}</p>
        </div>
        <button className="btn btn-primary" onClick={openCreate}>
          <i className="bi bi-plus-circle me-2"></i>Add Banner
        </button>
      </div>

      <div className="page-card">
        {loading ? (
          <div className="p-5 text-center"><div className="spinner-border text-primary"></div></div>
        ) : banners.length === 0 ? (
          <div className="empty-state">
            <i className="bi bi-image d-block mb-2" style={{ fontSize: '2rem', color: '#adb5bd' }}></i>
            <p>No banners yet</p>
            <button className="btn btn-primary btn-sm" onClick={openCreate}>Add First Banner</button>
          </div>
        ) : (
          <div className="table-responsive">
            <table className="admin-table">
              <thead>
                <tr><th>#</th><th>Preview</th><th>Title</th><th>Position</th><th>Link</th><th>Status</th><th>Actions</th></tr>
              </thead>
              <tbody>
                {banners.map(b => (
                  <tr key={b.id}>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{b.id}</td>
                    <td>
                      {b.image && (
                        <img src={b.image} style={{ width: 80, height: 44, objectFit: 'cover', borderRadius: 6 }}
                          onError={e => { e.target.style.display = 'none' }} alt="" />
                      )}
                    </td>
                    <td>
                      <div style={{ fontWeight: 600, fontSize: '.875rem' }}>{b.title || '(no title)'}</div>
                      {b.subtitle && <div className="text-muted" style={{ fontSize: '.75rem' }}>{b.subtitle}</div>}
                    </td>
                    <td>
                      {(() => {
                        const pos = POSITIONS.find(p => p.value === b.position)
                        if (!pos) return <span className="badge bg-secondary">{b.position}</span>
                        const [page, section] = pos.label.split('—').map(s => s.trim())
                        const sectionName = section ? section.replace(/\s*\(.*\)/, '').trim() : page
                        const colorMap = { home_top: 'primary', home_middle: 'warning', home_bottom: 'success', shop_top: 'info', category: 'secondary' }
                        return (
                          <div>
                            <div><span className={`badge bg-${colorMap[b.position] || 'light'} bg-opacity-10 text-${colorMap[b.position] || 'dark'}`} style={{ fontSize: '.7rem' }}>{sectionName}</span></div>
                            <div className="text-muted" style={{ fontSize: '.7rem' }}>{page}</div>
                          </div>
                        )
                      })()}
                    </td>
                    <td className="text-muted" style={{ fontSize: '.8rem' }}>{b.link || '—'}</td>
                    <td><span className={`badge-status badge-${b.is_active ? 'active' : 'inactive'}`}>{b.is_active ? 'Active' : 'Inactive'}</span></td>
                    <td>
                      <div className="action-btns">
                        <button className="btn btn-sm btn-outline-primary" onClick={() => openEdit(b)}><i className="bi bi-pencil"></i></button>
                        <button className="btn btn-sm btn-outline-danger" onClick={() => setConfirm(b.id)}><i className="bi bi-trash"></i></button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  )
}
export default Banners
