import { useRef, useState } from 'react'
import toast from 'react-hot-toast'
import { adminUploadImage } from '../../services/api'

const ImageUpload = ({ value, onChange, onRemove, label, required = false, height = 140 }) => {
  const [uploading, setUploading] = useState(false)
  const inputRef = useRef()

  const upload = async (file) => {
    if (!file) return
    if (!file.type.startsWith('image/')) { toast.error('Please select an image file'); return }
    if (file.size > 5 * 1024 * 1024) { toast.error('Image must be under 5MB'); return }
    setUploading(true)
    try {
      const res = await adminUploadImage(file)
      onChange(res.data.url)
      toast.success('Image uploaded!')
    } catch {
      toast.error('Upload failed')
    } finally {
      setUploading(false)
    }
  }

  const handleDrop = (e) => {
    e.preventDefault()
    e.stopPropagation()
    const file = e.dataTransfer.files?.[0]
    if (file) upload(file)
  }

  return (
    <div>
      {label && (
        <label className="form-label fw-semibold">
          {label}{required && ' *'}
        </label>
      )}
      <div
        onClick={() => !uploading && inputRef.current?.click()}
        onDrop={handleDrop}
        onDragOver={e => { e.preventDefault(); e.stopPropagation() }}
        style={{
          width: '100%', height,
          border: '2px dashed #dee2e6', borderRadius: 8,
          display: 'flex', alignItems: 'center', justifyContent: 'center',
          background: value ? '#fff' : '#fafafa',
          cursor: uploading ? 'default' : 'pointer',
          overflow: 'hidden', position: 'relative',
        }}
      >
        {uploading ? (
          <div className="text-center">
            <div className="spinner-border text-primary mb-2"></div>
            <div className="text-muted" style={{ fontSize: 12 }}>Uploading...</div>
          </div>
        ) : value ? (
          <>
            <img
              src={value} alt="preview"
              style={{ width: '100%', height: '100%', objectFit: 'contain' }}
              onError={e => { e.target.style.display = 'none' }}
            />
            <div style={{
              position: 'absolute', inset: 0, background: 'rgba(0,0,0,0.45)',
              display: 'flex', flexDirection: 'column', alignItems: 'center', justifyContent: 'center',
              opacity: 0, transition: 'opacity .2s',
            }}
              onMouseEnter={e => e.currentTarget.style.opacity = 1}
              onMouseLeave={e => e.currentTarget.style.opacity = 0}
            >
              <i className="bi bi-upload text-white" style={{ fontSize: 26 }}></i>
              <div className="text-white mt-1" style={{ fontSize: 12 }}>Click or drag to change</div>
            </div>
          </>
        ) : (
          <div className="text-center text-muted">
            <i className="bi bi-cloud-upload" style={{ fontSize: 34, color: '#adb5bd' }}></i>
            <div style={{ fontSize: 13, marginTop: 6 }}>Click or drag &amp; drop to upload</div>
            <div style={{ fontSize: 11, marginTop: 2, color: '#adb5bd' }}>JPG, PNG, WebP · Max 5MB</div>
          </div>
        )}
      </div>
      <input
        type="file"
        accept="image/*"
        ref={inputRef}
        style={{ display: 'none' }}
        onChange={e => { upload(e.target.files?.[0]); e.target.value = '' }}
      />
      {value && (
        <div className="mt-2 d-flex align-items-center gap-2">
          <input
            className="form-control form-control-sm text-muted"
            value={value}
            onChange={e => onChange(e.target.value)}
            placeholder="Image URL"
            style={{ fontSize: 11, color: '#6c757d' }}
          />
          <button
            type="button"
            className="btn btn-sm btn-outline-secondary flex-shrink-0"
            style={{ fontSize: 11, padding: '2px 8px' }}
            onClick={() => { navigator.clipboard.writeText(value); toast.success('URL copied!') }}
            title="Copy URL"
          >
            <i className="bi bi-clipboard"></i>
          </button>
          <button
            type="button"
            className="btn btn-sm btn-outline-danger flex-shrink-0"
            style={{ fontSize: 11, padding: '2px 8px' }}
            onClick={() => onRemove ? onRemove() : onChange('')}
            title="Remove image"
          >
            <i className="bi bi-trash"></i>
          </button>
        </div>
      )}
    </div>
  )
}

export default ImageUpload
