import { useState, useEffect, useRef } from 'react'
import { Link, useLocation } from 'react-router-dom'
import { useAuth } from '../../context/AuthContext'
import { getStats } from '../../services/api'

const TITLES = {
  '/': 'Dashboard', '/products': 'Products', '/categories': 'Categories',
  '/vendors': 'Vendors', '/orders': 'Orders', '/users': 'Users',
  '/homepage': 'Home Page', '/sliders': 'Sliders', '/banners': 'Banners', '/settings': 'Settings',
  '/wholesale': 'Wholesale Applications', '/coupons': 'Coupons',
  '/reviews': 'Reviews', '/contacts': 'Contacts', '/newsletter': 'Newsletter',
}

const Topbar = ({ onToggle }) => {
  const { pathname } = useLocation()
  const { user } = useAuth()
  const [notes, setNotes] = useState([])
  const [bellOpen, setBellOpen] = useState(false)
  const bellRef = useRef()
  const base = '/' + pathname.split('/')[1]
  const title = TITLES[base] || 'Admin'

  useEffect(() => {
    const fetch = () => {
      getStats().then(r => {
        const d = r.data
        const list = []
        if (d.pending_orders > 0) list.push({ label: `${d.pending_orders} pending order${d.pending_orders > 1 ? 's' : ''}`, icon: 'bi-cart-check', color: 'warning', to: '/orders' })
        if (d.unread_contacts > 0) list.push({ label: `${d.unread_contacts} unread message${d.unread_contacts > 1 ? 's' : ''}`, icon: 'bi-envelope', color: 'danger', to: '/contacts' })
        if (d.pending_reviews > 0) list.push({ label: `${d.pending_reviews} pending review${d.pending_reviews > 1 ? 's' : ''}`, icon: 'bi-star', color: 'secondary', to: '/reviews' })
        if (d.pending_wholesale > 0) list.push({ label: `${d.pending_wholesale} wholesale application${d.pending_wholesale > 1 ? 's' : ''}`, icon: 'bi-building', color: 'info', to: '/wholesale' })
        setNotes(list)
      }).catch(() => {})
    }
    fetch()
    const id = setInterval(fetch, 60000)
    return () => clearInterval(id)
  }, [])

  // Close bell dropdown on outside click
  useEffect(() => {
    const handler = (e) => { if (bellRef.current && !bellRef.current.contains(e.target)) setBellOpen(false) }
    document.addEventListener('mousedown', handler)
    return () => document.removeEventListener('mousedown', handler)
  }, [])

  return (
    <div className="topbar">
      <div className="topbar-left">
        <button className="topbar-hamburger d-lg-none" onClick={onToggle} aria-label="Toggle menu">
          <i className="bi bi-list"></i>
        </button>
        <h6 className="mb-0">{title}</h6>
      </div>

      <div className="topbar-right">
        <span className="badge bg-success" style={{ fontSize: '.7rem' }}>Live</span>

        {/* Notification bell */}
        <div className="position-relative" ref={bellRef}>
          <button
            className="topbar-icon-btn"
            onClick={() => setBellOpen(o => !o)}
            aria-label="Notifications"
          >
            <i className="bi bi-bell"></i>
            {notes.length > 0 && (
              <span className="topbar-badge">{notes.length > 9 ? '9+' : notes.length}</span>
            )}
          </button>

          {bellOpen && (
            <div className="topbar-dropdown">
              <div className="topbar-dropdown-header">
                <span className="fw-semibold" style={{ fontSize: '.85rem' }}>Notifications</span>
                {notes.length > 0 && <span className="badge bg-danger">{notes.length}</span>}
              </div>
              {notes.length === 0 ? (
                <div className="topbar-dropdown-empty">
                  <i className="bi bi-check-circle text-success fs-4 d-block mb-2"></i>
                  All caught up!
                </div>
              ) : (
                notes.map((n, i) => (
                  <Link key={i} to={n.to} className="topbar-dropdown-item" onClick={() => setBellOpen(false)}>
                    <div className={`topbar-notif-icon text-${n.color}`}>
                      <i className={`bi ${n.icon}`}></i>
                    </div>
                    <span style={{ fontSize: '.8rem' }}>{n.label}</span>
                  </Link>
                ))
              )}
            </div>
          )}
        </div>

        {/* User */}
        <div className="d-flex align-items-center gap-2">
          <div className="topbar-avatar">
            {user?.name?.[0]?.toUpperCase()}
          </div>
          <span className="d-none d-md-inline" style={{ fontSize: '.875rem', fontWeight: 500, color: '#495057' }}>
            {user?.name}
          </span>
        </div>
      </div>
    </div>
  )
}
export default Topbar
