import * as React from 'react';
import { FunctionComponent, SVGProps } from 'react';
import { RenderableAxisProps, PresentationAttributesAdaptChildEvent, TickItem, AxisDomain, ScaleType, AxisDomainTypeInput, TickProp, BaseTickContentProps } from '../util/types';
import { defaultPolarRadiusAxisProps } from './defaultPolarRadiusAxisProps';
import { RequiresDefaultProps } from '../util/resolveDefaultProps';
import { ZIndexable } from '../zIndex/ZIndexLayer';
import { CustomScaleDefinition } from '../util/scale/CustomScaleDefinition';
type TickOrientation = 'left' | 'right' | 'middle';
export interface PolarRadiusAxisProps extends Omit<RenderableAxisProps, 'axisLine' | 'angle' | 'type' | 'tickSize' | 'domain' | 'scale' | 'tick'>, ZIndexable {
    /**
     * Determines how the axis line is drawn. Options:
     * - `true`: the axis line is drawn with default props;
     * - `false`: the axis line is not visible;
     * - `object`: passed as props to SVG `<line>` element representing the axis line.
     *
     * @example <PolarRadiusAxis axisLine={false} />
     * @example <PolarRadiusAxis axisLine={{ stroke: 'red', strokeWidth: 2 }} />
     * @defaultValue true
     */
    axisLine?: boolean | SVGProps<SVGLineElement>;
    /**
     * The angle of the whole axis: the line, ticks and labels, everything.
     *
     * This is different from other graphical elements where angle usually means
     * the angle of text. Here, it means the angle of everything.
     *
     * @defaultValue 0
     */
    angle?: number;
    /**
     * The type of axis.
     *
     * `category`: Treats data as distinct values.
     * Each value is in the same distance from its neighbors, regardless of their actual numeric difference.
     *
     * `number`: Treats data as continuous range.
     * Values that are numerically closer are placed closer together on the axis.
     *
     * `auto`: the type is inferred based on the chart layout.
     *
     * @defaultValue auto
     */
    type?: AxisDomainTypeInput;
    /**
     * The orientation of axis text.
     * @defaultValue right
     */
    orientation?: TickOrientation;
    /**
     * Specify the domain of axis when the axis is a number axis.
     *
     * If undefined, then the domain is calculated based on the data and dataKeys.
     *
     * The length of domain should be 2, and we will validate the values in domain.
     *
     * Each element in the array can be a number, 'auto', 'dataMin', 'dataMax', a string like 'dataMin - 20', 'dataMax + 100',
     * or a function that accepts a single argument and returns a number.
     *
     * If any element of domain is set to be 'auto', comprehensible scale ticks will be calculated, and the final domain of axis is generated by the ticks.
     * If a function, receives '[dataMin, dataMax]', and must return a computed domain as '[min, max]'.
     *
     * @example <PolarRadiusAxis type="number" domain={['dataMin', 'dataMax']} />
     * @example <PolarRadiusAxis type="number" domain={[0, 'dataMax']} />
     * @example <PolarRadiusAxis type="number" domain={['auto', 'auto']} />
     * @example <PolarRadiusAxis type="number" domain={[0, 'dataMax + 1000']} />
     * @example <PolarRadiusAxis type="number" domain={['dataMin - 100', 'dataMax + 100']} />
     * @example <PolarRadiusAxis type="number" domain={[dataMin => (0 - Math.abs(dataMin)), dataMax => (dataMax * 2)]} />
     * @example <PolarRadiusAxis type="number" domain={([dataMin, dataMax]) => { const absMax = Math.max(Math.abs(dataMin), Math.abs(dataMax)); return [-absMax, absMax]; }} />
     * @example <PolarRadiusAxis type="number" domain={[0, 100]} allowDataOverflow />
     */
    domain?: AxisDomain;
    /**
     * Scale function determines how data values are mapped to visual values.
     * In other words, decided the mapping between data domain and coordinate range.
     *
     * If undefined, or 'auto', the scale function is created internally according to the type of axis and data.
     *
     * You can define a custom scale, either as a string shortcut to a d3 scale, or as a complete scale definition object.
     *
     * @defaultValue auto
     * @example <PolarRadiusAxis scale="log" />
     * @example
     * import { scaleLog } from 'd3-scale';
     * const scale = scaleLog().base(Math.E);
     * <PolarRadiusAxis scale={scale} />
     */
    scale?: ScaleType | CustomScaleDefinition | CustomScaleDefinition<string> | CustomScaleDefinition<number> | CustomScaleDefinition<Date>;
    /**
     * The customized event handler of click on the ticks of this axis
     */
    onClick?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mousedown on the ticks of this axis
     */
    onMouseDown?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseup on the ticks of this axis
     */
    onMouseUp?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mousemove on the ticks of this axis
     */
    onMouseMove?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseover on the ticks of this axis
     */
    onMouseOver?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseout on the ticks of this axis
     */
    onMouseOut?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseenter on the ticks of this axis
     */
    onMouseEnter?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * The customized event handler of mouseleave on the ticks of this axis
     */
    onMouseLeave?: (data: any, index: number, e: React.MouseEvent) => void;
    /**
     * Allow the ticks of axis to be decimals or not.
     *
     * @defaultValue false
     */
    allowDecimals?: boolean;
    /**
     * @defaultValue 0
     */
    radiusAxisId?: string | number;
    /**
     * Defines how the individual label text is rendered.
     * This controls the settings for individual ticks; on a typical axis, there are multiple ticks, depending on your data.
     *
     * If you want to customize the overall axis label, use the `label` prop instead.
     *
     * Options:
     * - `false`: Do not render any tick labels.
     * - `true`: Render tick labels with default settings.
     * - `object`: An object of props to be merged into the internally calculated tick props.
     * - `ReactElement`: A custom React element to be used as the tick label.
     * - `function`: A function that returns a React element for custom rendering of tick labels.
     *
     * @defaultValue true
     */
    tick?: TickProp<BaseTickContentProps>;
    ticks?: ReadonlyArray<TickItem>;
    /**
     * Z-Index of this component and its children. The higher the value,
     * the more on top it will be rendered.
     * Components with higher zIndex will appear in front of components with lower zIndex.
     * If undefined or 0, the content is rendered in the default layer without portals.
     *
     * @since 3.4
     * @defaultValue 500
     * @see {@link https://recharts.github.io/en-US/guide/zIndex/ Z-Index and layers guide}
     */
    zIndex?: number;
}
type AxisSvgProps = Omit<PresentationAttributesAdaptChildEvent<any, SVGTextElement>, 'scale' | 'type'>;
export type Props = AxisSvgProps & PolarRadiusAxisProps;
type PropsWithDefaults = RequiresDefaultProps<Props, typeof defaultPolarRadiusAxisProps>;
export declare const PolarRadiusAxisWrapper: FunctionComponent<PropsWithDefaults>;
/**
 * @provides PolarLabelContext
 * @consumes PolarViewBoxContext
 */
export declare function PolarRadiusAxis(outsideProps: Props): React.JSX.Element;
export declare namespace PolarRadiusAxis {
    var displayName: string;
}
export {};
